package main

/*
 * Theremin coil designer
 * See file "CHANGE_LOG.txt" for details
*/

import (
	"flag"
	"os"
)

func main() {
	// do help if no args
	if len(os.Args) < 2 {
		help_cmd(false)
    } else {
		// parse subcommands
		cmd := os.Args[1]
		switch cmd {
			case "help", "-help", "-h", "/h" : help()
			case "lcf" : lcf()
			case "pvc" : pvc()
			case "wire" : wire()
			case "brooks" : brooks()
			case "solenoid" : solenoid()
			case "spiral" : spiral()
			case "donut" : donut()
			case "choke" : choke()
			case "xfmr" : xfmr()
			default : cmd_hint(cmd)
		}
	}
}  // end of main()


////////////////////
// main functions //
////////////////////

// show help
func help() {
	sub := flag.NewFlagSet("help", flag.ExitOnError)
	verbose := sub.Bool("v", false, "verbose mode")
	sub.Parse(os.Args[2:])
	//
	help_cmd(*verbose)
}

// calculate one given two
func lcf() {
	sub := flag.NewFlagSet("lcf", flag.ExitOnError)
	l_mh := sub.Float64("l", 0, "inductance (mH)")
	c_pf := sub.Float64("c", 0, "capacitance (pF)")
	f_khz := sub.Float64("f", 0, "frequency (kHz)")
	r_ohms := sub.Float64("r", 0, "resistance (Ohms)")
	sub.Parse(os.Args[2:])
	//
	lcf_cmd(*l_mh, *c_pf, *f_khz, *r_ohms)
}

// return pvc specs
func pvc() {
	sub := flag.NewFlagSet("pvc", flag.ExitOnError)
	inches := sub.Bool("i", false, "Inches")
	sub.Parse(os.Args[2:])
	//
	pvc_cmd(*inches)
}

// return wire specs
func wire() {
	sub := flag.NewFlagSet("wire", flag.ExitOnError)
	awg := sub.Float64("awg", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	sub.Parse(os.Args[2:])
	//
	wire_cmd(*awg, *wcd_mm, *build)
}

// calculate brooks coil
func brooks() {
	sub := flag.NewFlagSet("brooks", flag.ExitOnError)
	l_mh := sub.Float64("l", 0, "inductance (mH)")
	awg := sub.Float64("awg", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	wd_mm := sub.Float64("wd", 0, "wire OD (mm)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	wl_m := sub.Float64("wl", 0, "wire length (m)")
	f_khz := sub.Float64("f", 0, "frequency (kHz)")
	c_pf := sub.Float64("c", 0, "`capacitance` (pF)")
	n := sub.Int("n", 0, "turns")
	file := sub.String("file", "", "file name")
	sub.Parse(os.Args[2:])
	//
	brooks_cmd(*file, *l_mh, *awg, *wd_mm, *wcd_mm, *wl_m, *f_khz, *c_pf, *build, *n)
}

// calculate flat spiral coil
func spiral() {
	sub := flag.NewFlagSet("spiral", flag.ExitOnError)
	l_mh := sub.Float64("l", 0, "inductance (mH)")
	awg := sub.Float64("awg", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	wd_mm := sub.Float64("wd", 0, "wire OD (mm)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	wl_m := sub.Float64("wl", 0, "wire length (m)")
	f_khz := sub.Float64("f", 0, "frequency (kHz)")
	c_pf := sub.Float64("c", 0, "`capacitance` (pF)")
	fd_mm := sub.Float64("fd", 0, "form dia (mm)")
	n := sub.Int("n", 0, "turns")
	file := sub.String("file", "", "file name")
	sub.Parse(os.Args[2:])
	//
	coil_cmd(*file, *l_mh, *awg, *wd_mm, *wcd_mm, *wl_m, *f_khz, *c_pf, *fd_mm, 0, *build, 1, 1, *n)
}

// calculate single layer solenoid coil
func solenoid() {
	sub := flag.NewFlagSet("solenoid", flag.ExitOnError)
	l_mh := sub.Float64("l", 0, "inductance (mH)")
	awg := sub.Float64("awg", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	wd_mm := sub.Float64("wd", 0, "wire OD (mm)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	wl_m := sub.Float64("wl", 0, "wire length (m)")
	f_khz := sub.Float64("f", 0, "frequency (kHz)")
	c_pf := sub.Float64("c", 0, "`capacitance` (pF)")
	fd_mm := sub.Float64("fd", 0, "form dia (mm)")
	n := sub.Int("n", 0, "turns")
	file := sub.String("file", "", "file name")
	sub.Parse(os.Args[2:])
	//
	coil_cmd(*file, *l_mh, *awg, *wd_mm, *wcd_mm, *wl_m, *f_khz, *c_pf, *fd_mm, 0, *build, 0, 1, *n)
}

// calculate multi-layer donut coil
func donut() {
	sub := flag.NewFlagSet("donut", flag.ExitOnError)
	l_mh := sub.Float64("l", 0, "inductance (mH)")
	awg := sub.Float64("awg", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	wd_mm := sub.Float64("wd", 0, "wire OD (mm)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	wl_m := sub.Float64("wl", 0, "wire length (m)")
	f_khz := sub.Float64("f", 0, "frequency (kHz)")
	c_pf := sub.Float64("c", 0, "`capacitance` (pF)")
	fd_mm := sub.Float64("fd", 0, "form dia (mm)")
	ow := sub.Int("ow", 0, "donut width (turns)")
	n := sub.Int("n", 0, "turns")
	file := sub.String("file", "", "file name")
	sub.Parse(os.Args[2:])
	//
	coil_cmd(*file, *l_mh, *awg, *wd_mm, *wcd_mm, *wl_m, *f_khz, *c_pf, *fd_mm, 0, *build, *ow, 1, *n)
}

// calculate multi-donut choke coil
func choke() {
	sub := flag.NewFlagSet("choke", flag.ExitOnError)
	l_mh := sub.Float64("l", 0, "inductance (mH)")
	awg := sub.Float64("awg", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	wd_mm := sub.Float64("wd", 0, "wire OD (mm)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	wl_m := sub.Float64("wl", 0, "wire length (m)")
	f_khz := sub.Float64("f", 0, "frequency (kHz)")
	c_pf := sub.Float64("c", 0, "capacitance (pF)")
	fd_mm := sub.Float64("fd", 0, "form dia (mm)")
	oo_mm := sub.Float64("oo", 0, "donut offset (mm)")
	ow := sub.Int("ow", 0, "donut width (turns)")
	o := sub.Int("o", 0, "donuts")
	n := sub.Int("n", 0, "turns")
	file := sub.String("file", "", "file name")
	sub.Parse(os.Args[2:])
	//
	coil_cmd(*file, *l_mh, *awg, *wd_mm, *wcd_mm, *wl_m, *f_khz, *c_pf, *fd_mm, *oo_mm, *build, *ow, *o, *n)
}

// calculate 2 donut transformer
func xfmr() {
	sub := flag.NewFlagSet("xfmr", flag.ExitOnError)
	awg := sub.Float64("awg", 0, "AWG gauge")
	awg2 := sub.Float64("awg2", 0, "AWG gauge")
	build := sub.Int("b", 1, "build (1 or 2 coats)")
	build2 := sub.Int("b2", 1, "build (1 or 2 coats)")
	wd_mm := sub.Float64("wd", 0, "wire OD (mm)")
	wd2_mm := sub.Float64("wd2", 0, "wire 2 OD (mm)")
	wcd_mm := sub.Float64("wcd", 0, "wire Cu dia (mm)")
	wcd2_mm := sub.Float64("wcd2", 0, "wire 2 Cu dia (mm)")
	fd_mm := sub.Float64("fd", 0, "form dia (mm)")
	fd2_mm := sub.Float64("fd2", 0, "form 2 dia (mm)")
	ow := sub.Int("ow", 0, "donut width (turns)")
	ow2 := sub.Int("ow2", 0, "donut 2 width (turns)")
	oo_mm := sub.Float64("oo", 0, "donut 2 offset (mm)")
	n := sub.Int("n", 0, "turns")
	n2 := sub.Int("n2", 0, "turns 2")
	file := sub.String("file", "", "file name")
	sub.Parse(os.Args[2:])
	//
	xfmr_cmd(*file, *awg, *awg2, *wd_mm, *wd2_mm, *wcd_mm, *wcd2_mm, *fd_mm, *fd2_mm, *oo_mm, *build, *build2, *ow, *ow2, *n, *n2) 
}
