package main

/*
 * tcoil support functions
*/

import (
	"os"
	"strings"
	"fmt"
)

// OpenSCAD code to do a single loop
var scad_str = `
$fn = 50;
loop_fn = 8;
clip = false;
//clip = true;

// single loop
// wr = wire radius
// or = loop radius
// oz = loop z location
// clip = cutaway view
module loop(wr, or, oz, clip) { 
	module torus() { rotate_extrude() translate([or, 0, 0]) circle(r=wr, $fn=loop_fn); }
	translate([0,0,oz]){
		if (clip) {
			z = wr*2;
			xy = z + or*2;
			intersection(){
				torus();
				translate([0,xy/2,0]) cube([xy,xy,z], center=true);
			}
		} 
		else {
			torus();
		}
	}
}

`

// write trimmed string to file
func file_write_str(file, data string) {
	err := os.WriteFile(file, []byte(strings.TrimSpace(data)), 0644)
	if err != nil { error_exit("write to file " + file + " failed") }
}

// create OpenSCAD file to view choke (brooks, donut, choke)
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// N = turns total
func scad_choke(file, prn_str string, loop_r, wire_r, oo float64, ow, o, N int) {
	prn_str = "// file: " + file + "\n/*\n" + prn_str + "*/\n"
	prn_str += scad_str
	prn_str += "color(\"green\"){\n"
	for i:=0; i<N; i++ {
		r, x, _ := choke_rxo(loop_r, wire_r, oo, ow, o, i, N)
		prn_str += fmt.Sprint("loop(", wire_r, ", ", r, ", ", x, ", clip);\n")
	}
	prn_str += "}\n"
	file_write_str(file, prn_str)
	return
}

// create OpenSCAD file to view xfmr
// -inputs-
// loop_r, loop2_r = loop radius (m) @ centerline of conductors
// wire_r, wire2_r = wire radius (m)
// cond_r, cond2_r = conductor radius (m)
// ow, ow2 = donut width (turns per layer)
// oo = donut offset (m)
// N, N2 = turns
func scad_xfmr(file, prn_str string, loop_r, loop2_r, wire_r, wire2_r, cond_r, cond2_r, oo float64, ow, ow2, N, N2 int) {
	prn_str = "// file: " + file + "\n/*\n" + prn_str + "*/\n"
	prn_str += scad_str
	prn_str += "color(\"green\"){\n"
	for i:=0; i<N; i++ {
		r, x := donut_rx(loop_r, wire_r, ow, i)
		prn_str += fmt.Sprint("loop(", wire_r, ", ", r, ", ", x, ", clip);\n")
	}
	prn_str += "}\n"
	prn_str += "color(\"red\"){\n"
	for j:=0; j<N2; j++ {
		r2, x2 := donut_rx(loop2_r, wire2_r, ow2, j)
		prn_str += fmt.Sprint("loop(", wire2_r, ", ", r2, ", ", x2 + oo, ", clip);\n")
	}
	prn_str += "}\n"
	file_write_str(file, prn_str)
	return
}
