package main

/*
 * tcoil command functions
*/

import (
	"math"
	"fmt"
	"os"
	"strings"
)

// tcoil commands
var tcoil_cmds = []string {  
	"help",
	"wire",
	"lcf",
	"pvc",
	"brooks",
	"spiral",
	"solenoid",
	"donut",
	"choke",
	"xfmr",
}

// bad commmand, give hint if match, exit
func cmd_hint(cmd string) {
	str := fmt.Sprint("Command '", cmd, "' not found")
	list := fuzzy_list(cmd, tcoil_cmds, 1, true)
	if len(list) > 0 { str += " (did you mean '" + list[0] + "' ?)" }
	error_exit(str)
}

// print error message and exit program
func error_exit(error_str string) {
	fmt.Println("> -ERROR-", error_str, "!")
	os.Exit(0) 
}

// check for error, exit program if true
func err_chk(err error) {
	if err != nil { error_exit(err.Error()) }
}

// print help info
func help_cmd(verbose_f bool) {
	str := fmt.Sprintln("\n= TCOIL VERSION", VER, "=") 
	str += "Theremin Inductor Design Tool\n"
	str += help_commands_str
	if verbose_f { str += help_verbose_str }
	fmt.Println(strings.Replace(str, "TCOIL_EXE", os.Args[0], -1))
}

// do lcf
// given 2 solve for the 3rd
// l_mh = inductance (mH)
// c_pf = capacitance (pF)
// f_khz = frequency (kHz)
// r_ohms = DCR (Ohms)
func lcf_cmd(l_mh, c_pf, f_khz, r_ohms float64) {
	// cmd line feedback
	prn_str := "> lcf:"
	if l_mh != 0 { prn_str += fmt.Sprintf(" l[%0.4g]mH,", l_mh) }
	if c_pf != 0 { prn_str += fmt.Sprintf(" c[%0.4g]pF,", c_pf) }
	if f_khz != 0 { prn_str += fmt.Sprintf(" f[%0.4g]kHz,", f_khz) }
	if r_ohms != 0 { prn_str += fmt.Sprintf(" r[%0.4g]Ohms", r_ohms) }
	prn_str = strings.TrimSuffix(prn_str, ",")
	prn_str += "\n\n"
	// input testing
	if l_mh < 0 { error_exit("-l cannot be negative") }
	if c_pf < 0 { error_exit("-c cannot be negative") }
	if f_khz < 0 { error_exit("-f cannot be negative") }
	if r_ohms < 0 { error_exit("-r cannot be negative") }
	// input norm
	l := l_mh / 1e3
	c := c_pf / 1e12
	f := f_khz * 1e3
	// calcs
	switch {
	case l_mh != 0 && c_pf != 0 && f_khz == 0 :  // lc_to_f
		f = lc_f(l, c)
	case l_mh != 0 && c_pf == 0 && f_khz != 0 :  // fl_to_c
		c = fl_c(f, l)
	case l_mh == 0 && c_pf != 0 && f_khz != 0 :  // fc_to_l
		l = fc_l(f, c)
	default: error_exit("lcf: nothing to do")
	}
	// results
	prn_str += fmt.Sprintf("  %-3s%6.4g mH\n", "L", l * 1e3)
	prn_str += fmt.Sprintf("  %-3s%6.4g pF\n", "C", c * 1e12)
	prn_str += fmt.Sprintf("  %-3s%6.4g kHz\n", "F", f / 1e3)
	if r_ohms > 0 { 
		q := math.Sqrt(l / c) / r_ohms
		prn_str += fmt.Sprintf("  %-3s%6.4g Ohms\n", "R", r_ohms)
		prn_str += fmt.Sprintf("  %-3s%6.4g\n", "Q", q)
	}
	fmt.Println(prn_str)
}

// do pvc stats
func pvc_cmd(inches bool) {
	// cmd line feedback
	prn_str := "> pvc: "
	if inches { prn_str += fmt.Sprint("i") }
	prn_str += "\n\n"
	prn_str += pvc_table(inches)
	fmt.Println(prn_str)
}

// do wire stats
func wire_cmd(awg, wcd_mm float64, build int) {
	// cmd line feedback
	prn_str := "> wire:"
	if awg != 0 { prn_str += fmt.Sprint(" awg[", awg, "],") }
	if wcd_mm != 0 { prn_str += fmt.Sprint(" wcd[", wcd_mm, "]mm,") }
	if build != 1 { prn_str += fmt.Sprint(" b[", build, "],") }
	prn_str = strings.TrimSuffix(prn_str, ",")
	prn_str += "\n\n"
	// input testing
	if awg < 0 { error_exit("-awg cannot be negative") }
	if wcd_mm < 0 { error_exit("-wcd cannot be negative") }
	// calcs
	switch {
	case awg != 0 && wcd_mm == 0 :  // do awg
		wcd_mm = awg_to_wcd(awg)
	case awg == 0 && wcd_mm != 0 :  // do wcd_mm
		awg = wcd_to_awg(wcd_mm)
	default: error_exit("wire: nothing to do")
	}
	// results
	wd_mm := build_dia(wcd_mm, build)
	wcr_m := wcd_mm / 2e3  // conductor dia mm => rad m
	dcr := dcr_per_m(wcr_m)
	prn_str += fmt.Sprintf("  %-10s%6.3g\n", "AWG", awg)
	prn_str += fmt.Sprintf("  %-10s%6d\n", "Build", build)
	prn_str += fmt.Sprintf("  %-10s%6.4g mm\n", "Cu Dia", wcd_mm)
	prn_str += fmt.Sprintf("  %-10s%6.4g mm\n", "OD", wd_mm)
	prn_str += fmt.Sprintf("  %-10s%6.4g kHz\n", "Skin", skin_freq(wcr_m) / 1e3)
	prn_str += fmt.Sprintf("  %-10s%6.4g Ohms/m\n", "DCR", dcr)
	fmt.Println(prn_str)
}

// do multi-donut choke, single donut, single layer solenoid, etc.
func coil_cmd(file string, l_mh, awg, wd_mm, wcd_mm, wl_m, f_khz, c_pf, fd_mm, oo_mm float64, build, ow, o, n int) {
	// cmd line feedback
	mode := os.Args[1]
	prn_str := "> " + mode + ":"
	if awg != 0 { prn_str += fmt.Sprint(" awg[", awg, "],") }
	if build != 1 { prn_str += fmt.Sprint(" b[", build, "],") }
	if wd_mm != 0 { prn_str += fmt.Sprint(" wd[", wd_mm, "]mm,") }
	if wcd_mm != 0 { prn_str += fmt.Sprint(" wcd[", wcd_mm, "]mm,") }
	if mode != "brooks" {
		if fd_mm != 0 { prn_str += fmt.Sprint(" fd[", fd_mm, "]mm,") }
	}
	if mode == "choke" || mode == "donut" {
		if ow != 0 { prn_str += fmt.Sprint(" ow[", ow, "]turns,") }
	}
	if mode == "choke" {
		if o != 1 { prn_str += fmt.Sprint(" o[", o, "]donuts,") }
		if oo_mm != 0 { prn_str += fmt.Sprint(" oo[", oo_mm, "]mm,") }
	}
	if f_khz != 0 { prn_str += fmt.Sprint(" f[", f_khz, "]kHz,") }
	if c_pf != 0 { prn_str += fmt.Sprint(" c[", c_pf, "]pF,") }
	if l_mh != 0 { prn_str += fmt.Sprint(" l[", l_mh, "]mH,") }
	if n != 0 { prn_str += fmt.Sprint(" n[", n, "]turns,") }
	if wl_m != 0 { prn_str += fmt.Sprint(" wl[", wl_m, "]m,") }
	if file != "" { prn_str += fmt.Sprint(" file[", file, "],") }
	prn_str = strings.TrimSuffix(prn_str, ",")
	// input testing
	if c_pf < 0 { error_exit("-c cannot be negative") }
	if f_khz < 0 { error_exit("-f cannot be negative") }
	if f_khz > 0 && c_pf > 0 { error_exit("-f, -c: please pick only one") }
	//
	if awg > 0 && wcd_mm > 0 { error_exit("-awg, -wcd: please pick only one") }
	if awg > 0 {
		wcd_mm = awg_to_wcd(awg)
		wd_mm = build_dia(wcd_mm, build)
	} else {
		awg = wcd_to_awg(wcd_mm)
	}
	if awg < 0 { error_exit("-awg cannot be negative") }
	if wd_mm <= 0 { error_exit("-wd must be greater than zero") }
	if wcd_mm <= 0 { error_exit("-wcd must be greater than zero") }
	if wcd_mm > wd_mm { error_exit("-wcd cannot be greater than -wd") }
	if fd_mm <= 0 { error_exit("-fd must be greater than zero") }
	if o <= 0 { error_exit("-o must be greater than zero") }
	if ow < 0 { error_exit("-ow cannot be negative") }
	// input norm
	l_h := l_mh / 1e3  // target L mH => H
	fr_m := fd_mm / 2e3  // former dia mm => rad m
	wr_m := wd_mm / 2e3  // wire dia mm => rad m
	wcr_m := wcd_mm / 2e3  // conductor dia mm => rad m
	lr_m := fr_m + wr_m  // to centerline of loop wire
	oo_m := oo_mm / 1e3  // donuts offset
	l_mut := 0.0
	l_self := 0.0
	// calcs
	switch {
	case wl_m > 0 && n == 0 && l_mh == 0 :  // wl => N
		n = choke_wl_N(lr_m, wr_m, wl_m, oo_m, ow, o)
	case wl_m == 0 && n > 0 && l_mh == 0 :  // N => wl
		wl_m = choke_N_wl(lr_m, wr_m, oo_m, ow, o, n)
	case wl_m == 0 && n == 0 && l_mh > 0 :  // L => N => wl
		n = choke_L_N(l_h, lr_m, wr_m, wcr_m, oo_m, ow, o)
		wl_m = choke_N_wl(lr_m, wr_m, oo_m, ow, o, n)
	default: error_exit(mode + ": nothing to do")
	}
	prn_str += "\n\n"
	// N => L
	l_mut, l_self = choke_N_L(lr_m, wr_m, wcr_m, oo_m, ow, o, n, true)
	l_total := l_mut + l_self
	// dimensional stuff
	r_max, x_max := choke_rx_max(lr_m, wr_m, oo_m, ow, o, n)
	wd_m := 2 * wr_m
	on := float64(n) / float64(o)  // turns per donut
	if ow == 0 { ow = choke_on(o, n) }  // single layer donuts
	ow_m := float64(ow) * wd_m
	layers := 1 + int(math.Round((r_max - lr_m) / wd_m))
	dia_m := 2 * (r_max + wr_m)
	width_m := x_max + wd_m
	height_m := dia_m / 2 - fr_m
	// frequency stuff
	c_f := c_pf * 1e-12
	if c_pf > 0 { 
		f_khz = lc_f(l_total, c_f) * 1e-3 
	} else {
		c_f = fl_c(f_khz * 1e3, l_total)
	}
	kg := kg_per_m(wcr_m) * wl_m
	dcr := dcr_per_m(wcr_m) * wl_m
	f_hz := f_khz * 1e3
	skin := skin_factor(wcr_m, f_hz)
	prox := choke_prox(lr_m, wr_m, wcr_m, oo_m, f_hz, ow, o, n)
	acr := dcr * prox * skin
	Q := lcr_q(l_total, c_f, acr)
	//
	prn_str += fmt.Sprintf("  %-12s%10s\n", "Type", strings.ToUpper(mode))
	prn_str += fmt.Sprintf("  %-12s%10.4g mH\n", "L total", l_total * 1e3)
	prn_str += fmt.Sprintf("  %-12s%10d turns\n", "N total", n)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g m\n", "Wire Length", wl_m)
	prn_str += fmt.Sprintf("  %-12s%10.4g g\n", "Wire Mass", kg*1e3)
	prn_str += fmt.Sprintf("  %-12s%10.3g\n", "Wire AWG", awg)
	prn_str += fmt.Sprintf("  %-12s%10.4g mm\n", "Wire Cu Dia", wcd_mm)
	prn_str += fmt.Sprintf("  %-12s%10.4g mm\n", "Wire OD", wd_mm)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g mm\n", "Form Dia", fd_mm)
	prn_str += fmt.Sprintf("  %-12s%10.4g mm\n", "Turns OD", dia_m * 1e3)
	if o == 1 { //  single donut
		prn_str += fmt.Sprintf("  %-12s%10.4g mm (%d)\n", "Turns Height", height_m * 1e3, layers) 
		prn_str += fmt.Sprintf("  %-12s%10.4g mm (%d)\n", "Turns Width", ow_m * 1e3, ow)
		prn_str += fmt.Sprintf("  %-12s%10.4g (x-section)\n", "Turns H/W", height_m / ow_m)
		prn_str += fmt.Sprintf("  %-12s%10.4g (aspect)\n", "Turns W/OD", width_m / dia_m)
	} else {  // multiple donuts
		prn_str += fmt.Sprintf("  %-12s%10.4g mm\n", "Turns Width", width_m * 1e3)
		prn_str += fmt.Sprintf("  %-12s%10.4g (aspect)\n", "Turns W/OD", width_m / dia_m)
		prn_str += "\n"
		prn_str += fmt.Sprintf("  %-12s%10d\n", "Donuts", o)
		prn_str += fmt.Sprintf("  %-12s%10.4g turns\n", "Donut N", on)
		prn_str += fmt.Sprintf("  %-12s%10.4g mm (%d)\n", "Donut Height", height_m * 1e3, layers)
		prn_str += fmt.Sprintf("  %-12s%10.4g mm (%d)\n", "Donut Width", ow_m * 1e3, ow)
		prn_str += fmt.Sprintf("  %-12s%10.4g (x-section)\n", "Donut H/W", height_m / ow_m)
		gap := choke_gap(lr_m, wr_m, wcr_m, oo_m, ow, o, n)
		prn_str += fmt.Sprintf("  %-12s%10.4g mm", "Donut Gap", gap * 1e3)
		if gap < -OVERLAP { prn_str += " (COLLISION!)" }
		prn_str += "\n"
	}
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g mH\n", "L Mutual", 1e3 * l_mut)
	prn_str += fmt.Sprintf("  %-12s%10.4g mH\n", "L Self", 1e3 * l_self)
	prn_str += fmt.Sprintf("  %-12s%10.4g\n", "L M/S", l_mut/l_self)
	prn_str += fmt.Sprintf("  %-12s%10.4g uH/m\n", "L/Wire Len", 1e6 * l_total / wl_m)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g Ohms\n", "DCR", dcr)
	if f_khz > 0 || c_pf > 0 {  // ac stuff
		prn_str += fmt.Sprintf("  %-12s%10.4g\n", "Skin", 1 / skin)
		prn_str += fmt.Sprintf("  %-12s%10.4g\n", "Proximity", 1 / prox)
		prn_str += fmt.Sprintf("  %-12s%10.4g Ohms @ %0.4g kHz\n", "ACR", acr, f_khz)
		if c_pf > 0 { prn_str += fmt.Sprintf("  %-12s%10.4g (CAUTION: NO SELF C!)\n", "LCR Q", Q)
		} else { prn_str += fmt.Sprintf("  %-12s%10.4g @ %0.4g kHz (CAUTION: NO SELF C!)\n", "LR Q", Q, f_khz) }
	}
	if file != "" {
		file += ".scad"
		scad_choke(file, prn_str, lr_m, wr_m, oo_m, ow, o, n)
		prn_str += "\n> Wrote to file: " + file + "\n"
	}
	fmt.Println(prn_str)
}

// do brooks coil
// based on formulas: 
// L = 2.034 * u0 * c * N^2
// where c = winding width, height, ID radius
// c = wd * sqrt(N)
// N = [L / (2.034 * u0 * wd)] ^ (2/5)
// N = [wl / (3 * pi * wd)] ^ (2/3)
func brooks_cmd(file string, l_mh, awg, wd_mm, wcd_mm, wl_m, f_khz, c_pf float64, build, n int) {
	wd_m := wd_mm / 1e3
	if awg > 0 { 
		wd_m = build_dia(awg_to_wcd(awg), build) / 1e3 
	}
	// input testing
	if wd_m <= 0 { error_exit("-wd must be greater than zero") }
	// input norm
	l_h := l_mh / 1e3
	n_flt := float64(n)
	// calcs
	switch {
	case wl_m > 0 && n == 0 && l_mh == 0 :  // wl => N
		n_flt = math.Pow((wl_m / (3 * math.Pi * wd_m)), 2.0/3.0)
	case wl_m == 0 && n > 0 && l_mh == 0 :  // N => wl
		// no calcs necessary
	case wl_m == 0 && n == 0 && l_mh > 0 :  // L => N => wl
		n_flt = math.Pow(l_h / (2.034 * MU0 * wd_m), 2.0/5.0)
	default: error_exit("brooks: nothing to do")
	}
	n_sqrt := math.Sqrt(n_flt)
	ow := int(math.Round(n_sqrt))
	fd_mm := n_sqrt * wd_m * 2e3
	coil_cmd(file, l_mh, awg, wd_mm, wcd_mm, wl_m, f_khz, c_pf, fd_mm, 0, build, ow, 1, n)
}

// do transformer
func xfmr_cmd(file string, awg, awg2, wd_mm, wd2_mm, wcd_mm, wcd2_mm, fd_mm, fd2_mm, oo_mm float64, build, build2, ow, ow2, n, n2 int) {
	// cmd line feedback
	prn_str := "> xfmr:"
	prn_str += fmt.Sprint(" n[", n, "]turns,") 
	if awg != 0 { prn_str += fmt.Sprint(" awg[", awg, "],") }
	if build != 1 { prn_str += fmt.Sprint(" b[", build, "],") }
	if wd_mm != 0 { prn_str += fmt.Sprint(" wd[", wd_mm, "]mm,") }
	if wcd_mm != 0 { prn_str += fmt.Sprint(" wcd[", wcd_mm, "]mm,") }
	prn_str += fmt.Sprint(" fd[", fd_mm, "]mm,")
	if ow != 0 { prn_str += fmt.Sprint(" ow[", ow, "]turns,") }
	if oo_mm != 0 { prn_str += fmt.Sprint(" oo[", oo_mm, "]mm,") }
	// do 0 input defaults
	if n2 != 0 { prn_str += fmt.Sprint(" n2[", n2, "]turns,") } else { n2 = n }
	if awg2 != 0 { prn_str += fmt.Sprint(" awg2[", awg2, "],") } else { awg2 = awg }
	if build2 != 1 { prn_str += fmt.Sprint(" b2[", build2, "],") } else { build2 = build }
	if wd2_mm != 0 { prn_str += fmt.Sprint(" wd2[", wd2_mm, "]mm,") } else { wd2_mm = wd_mm }
	if wcd2_mm != 0 { prn_str += fmt.Sprint(" wcd2[", wcd2_mm, "]mm,") } else { wcd2_mm = wcd_mm }
	if fd2_mm != 0 { prn_str += fmt.Sprint(" fd2[", fd2_mm, "]mm,") } else { fd2_mm = fd_mm }
	if ow2 != 0 { prn_str += fmt.Sprint(" ow2[", ow2, "]turns,") } else { ow2 = ow }
	if file != "" { prn_str += fmt.Sprint(" file[", file, "],") }
	prn_str = strings.TrimSuffix(prn_str, ",")
	prn_str += "\n\n"
	// input testing
	if awg > 0 && wcd_mm > 0 { error_exit("-awg, -wcd: please pick only one") }
	if awg > 0 {
		wcd_mm = awg_to_wcd(awg)
		wd_mm = build_dia(wcd_mm, build)
	} else {
		awg = wcd_to_awg(wcd_mm)
	}
	if awg < 0 { error_exit("-awg cannot be negative") }
	if wd_mm <= 0 { error_exit("-wd must be greater than zero") }
	if wcd_mm <= 0 { error_exit("-wcd must be greater than zero") }
	if wcd_mm > wd_mm { error_exit("-wcd cannot be greater than -wd") }
	if fd_mm <= 0 { error_exit("-fd must be greater than zero") }
	if ow < 0 { error_exit("-ow cannot be negative") }
	if n <= 0 { error_exit("-n must be greater than zero") }
	//
	if awg2 > 0 && wcd2_mm > 0 { error_exit("-awg2, -wcd2: please pick only one") }
	if awg2 > 0 {
		wcd2_mm = awg_to_wcd(awg2)
		wd2_mm = build_dia(wcd2_mm, build2)
	} else {
		awg2 = wcd_to_awg(wcd2_mm)
	}
	if awg2 < 0 { error_exit("-awg2 cannot be negative") }
	if wd2_mm <= 0 { error_exit("-wd2 must be greater than zero") }
	if wcd2_mm <= 0 { error_exit("-wcd2 must be greater than zero") }
	if wcd2_mm > wd2_mm { error_exit("-wcd2 cannot be greater than -wd2") }
	if fd2_mm < 0 { error_exit("-fd2 cannot be negative") }
	if ow2 < 0 { error_exit("-ow2 cannot be negative") }
	if n2 <= 0 { error_exit("-n2 must be greater than zero") }
	// input norm
	wr_m := wd_mm / 2e3  // wire dia mm => rad m
	wcr_m := wcd_mm / 2e3  // conductor dia mm => rad m
	fr_m := fd_mm / 2e3  // form dia mm => rad m
	//
	wr2_m := wd2_mm / 2e3  // wire dia mm => rad m
	wcr2_m := wcd2_mm / 2e3  // conductor dia mm => rad m
	fr2_m := fd2_mm / 2e3  // form dia mm => rad m
	oo_m := oo_mm / 1e3  // donut offset
	// calcs
	lr_m := fr_m + wr_m
	r_max, x_max := choke_rx_max(lr_m, wr_m, 0, ow, 1, n)
	wd_m := 2 * wr_m
	layers := 1 + int(math.Round((r_max - lr_m) / wd_m))
	dia_m := 2 * (r_max + wr_m)
	width_m := x_max + wd_m
	height_m := float64(layers) * wd_m
	//
	lr2_m := fr2_m + wr2_m
	r2_max, x2_max := choke_rx_max(lr2_m, wr2_m, 0, ow2, 1, n2)
	wd2_m := 2 * wr2_m
	layers2 := 1 + int(math.Round((r2_max - lr2_m) / wd2_m))
	dia2_m := 2 * (r2_max + wr2_m)
	width2_m := x2_max + wd2_m
	height2_m := float64(layers2) * wd2_m
	//
	l_mut, l_self := choke_N_L(lr_m, wr_m, wcr_m, 0, ow, 1, n, false)
	l_total := l_mut + l_self
	//
	l2_mut, l2_self := choke_N_L(lr2_m, wr2_m, wcr2_m, 0, ow2, 1, n2, false)
	l2_total := l2_mut + l2_self
	//
	lx_mut := xfmr_N_L(lr_m, lr2_m, wr_m, wr2_m, wcr_m, wcr2_m, oo_m, ow, ow2, n, n2)
	k := lx_mut / (2 * math.Sqrt(l_total * l2_total))
	//
	wl_m := choke_N_wl(lr_m, wr_m, 0, ow, 1, n)
	wl2_m := choke_N_wl(lr2_m, wr2_m, 0, ow2, 1, n2)
	//
	kg := kg_per_m(wcr_m) * wl_m
	kg2 := kg_per_m(wcr2_m) * wl2_m
	//
	dcr := dcr_per_m(wcr_m) * wl_m
	dcr2 := dcr_per_m(wcr2_m) * wl2_m
	// clearances
	gap := xfmr_gap(lr_m, lr2_m, wr_m, wr2_m, wcr_m, wcr2_m, oo_m, ow, ow2, n, n2)
	//
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mH\n", "L Total", l_total*1e3, l2_total*1e3)
	prn_str += fmt.Sprintf("  %-12s%10d%10d turns\n", "N Total", n, n2)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g m\n", "Wire Length", wl_m, wl2_m)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g g\n", "Wire Mass", kg*1e3, kg2*1e3)
	prn_str += fmt.Sprintf("  %-12s%10.3g%10.3g\n", "Wire AWG", awg, awg2)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mm\n", "Wire Cu Dia", wcd_mm, wcd2_mm)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mm\n", "Wire OD", wd_mm, wd2_mm)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mm\n", "Form Dia", fd_mm, fd2_mm)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mm\n", "Turns OD", dia_m*1e3, dia2_m*1e3)
	prn_str += fmt.Sprintf("  %-12s%10d%10d\n", "Turns Layers", layers, layers2)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mm\n", "Turns Height", height_m*1e3, height2_m*1e3)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mm\n", "Turns Width", width_m*1e3, width2_m*1e3)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g (x-section)\n", "Turns H/W", height_m/width_m, height2_m/width2_m)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g (aspect)\n", "Turns W/OD", width_m/dia_m, width2_m/dia2_m)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g uH/m\n", "L/Wire Len", 1e6*l_total/wl_m, 1e6*l2_total/wl2_m)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mH\n", "L Mutual", 1e3 * l_mut, 1e3 * l2_mut)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g mH\n", "L Self", 1e3 * l_self, 1e3 * l2_self)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g\n", "L M/S", l_mut/l_self, l2_mut/l2_self)
	prn_str += fmt.Sprintf("  %-12s%10.4g%10.4g Ohms\n", "DCR", dcr, dcr2)
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g mm", "Gap", gap * 1e3)
	if gap < -OVERLAP { prn_str += " (COLLISION!)" }
	prn_str += "\n"
	prn_str += fmt.Sprintf("  %-12s%10.4g mH\n", "LX Mutual", lx_mut * 1e3)
	prn_str += fmt.Sprintf("  %-12s%10.4g mH\n", "L Total", (lx_mut + l_total + l2_total) * 1e3)
	prn_str += fmt.Sprintf("  %-12s%10.4g\n", "k", k)
	if file != "" {
		file += ".scad"
		scad_xfmr(file, prn_str, lr_m, lr2_m, wr_m, wr2_m, wcr_m, wcr2_m, oo_m, ow, ow2, n, n2)
		prn_str += "\n> Wrote to file: " + file + "\n"
	}
	fmt.Println(prn_str)
}


