package main

/*
 * tcoil LCR type functions
*/

import (
	"math"
	"fmt"
)

////////////////
// L, C, F, Q //
////////////////

// given L(H) & C(F) find freq(Hz)
func lc_f(l, c float64) (float64) {
	return (1 / (2 * math.Pi * math.Sqrt(l * c)))
}

// given freq(Hz) & L(H) find C(F)
func fl_c(f, l float64) (float64) {
	return ((math.Pow(1 / (2 * math.Pi * f), 2) / l))
}

// given freq(Hz) & L(H) find C(F)
func fc_l(f, c float64) (float64) {
	return ((math.Pow(1 / (2 * math.Pi * f), 2) / c))
}

// return quality factor Q
// given L(H), C(F), R(Ohms)
// Q = 1/R * sqrt(L/C)
func lcr_q(l, c, r float64) (q float64) {
	q = (1 / r) * math.Sqrt(l / c)
	return
}


//////////////////////////
// Elliptical integrals //
//////////////////////////


// given k, calculate the complete elliptic integrals:
// K(k) = first kind
// E(k) = second kind
// Reference: "Pi and the AGM" by J. Borwein and P. Borwein.
// The calculation here is based in the arithmetic geometric mean (agm) where
//   K(k) = pi / (2*agm(1, sqrt(1 - k^2))
// To compute the agm we use two interdependent sequences:
//   a(n+1) = ((a(n) + g(n))/2 = arithmetic mean
//   g(n+1) = sqrt(a(n) * g(n)) = geometric mean
// which begins as:
//   a(0) = 1
//   g(0) = sqrt(1 - k^2)
// a and g converge quadratically, stopping when |a - b| < EPSILON
//
// For E(k):
//   c(n+1) = c(n) - i(n) * ((a(n) - g(n))/2)^2
//   i(n+1) = 2*i(n)
// where:
//   c(0) = 1 - (k^2)/2
//   i(0) = 1
// then:
//   E(k) = c * K(k)
// which we calculate in parallel
func elliptic_K_E(k float64) (K, E float64) {
	c := 1 - k * k / 2
	i := 1.0
	//
	a := 1.0
	g := math.Sqrt(1 - k * k)
	for (math.Abs(a - g) > EPSILON) {
		c -= i * math.Pow((a - g) / 2, 2)
		i *= 2
		// update a & g
		a_nxt := (a + g) / 2
		g = math.Sqrt(a * g)
		a = a_nxt
	}
	K = math.Pi / (2 * a)
	E = c * K
	return
}


////////////////////////
// Loop self & mutual //
////////////////////////

// find mutual inductance between two coaxial loops of wire
// via complete elliptic integrals K(k) & E(k)
// from: "Classical Calculation for Mutual Inductance of Two Coaxial Loops in MKS Units" by Kurt Nalty
// -inputs-
// loop_r, loop2_r = loop radii (m) @ centerline of conductors
// dist = distance between loops (m) @ centerline of conductors
// -outputs-
// l_mut = mutual inductance (H)
func loops_l_mut(loop_r, loop2_r, dist float64) (l_mut float64) {
	r1_sq := loop_r * loop_r
	r2_sq := loop2_r * loop2_r
	d_sq := dist * dist
	a := (r1_sq + r2_sq + d_sq) / (r1_sq * r2_sq)
	b := 2 / (loop_r * loop2_r)
	beta := math.Sqrt(2 * b / (a + b))
	K, E := elliptic_K_E(beta)
	l_mut = 2 * MU0 * (math.Sqrt(a + b) / b) * ((1 - beta * beta / 2) * K - E)
	return
}

// find the self L of single loop w/ round conductor
// by computing the mutual inductance between two filamental conductors 
// placed at a distance equal to the geometrical mean distance (gmd) of every 
// pair of points in the section of the conductor. 
// from: "Mutual Inductance and Inductance Calculations by Maxwell’s Method" by Antonio Carlos M. de Queiroz
// Some gmd's:
// R = r * e^(-1/4)  // round wire of cross sectional radius r
// R = a * e^(-3/2)  // flat wire of width a
// R = e^(log(b) + (1/3)*log(2) + pi/3 - 25/12) // square wire with side b
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// cond_r = conductor radius (m)
// -outputs-
// l_self = self inductance (H)
func loop_l_self(loop_r, cond_r float64) (l_self float64) {
	R := cond_r * math.Exp(-1.0 / 4.0)
	l_self = loops_l_mut(loop_r, loop_r, R)
	return
}


///////////
// Choke //
///////////

// return r & x for loop n in multi-layer donut
// wires are laid down per layer
// if ow = 0 do single layer
// -inputs-
// loop_r = loop radius (m) of first layer @ centerline of conductor
// wire_r = wire radius (m) of OD
// ow = donut width (turns per layer)
// n = loop number
// -outputs-
// r = radius of loop (m)
// x = location of loop (m)
func donut_rx(loop_r, wire_r float64, ow, n int) (r, x float64) {
	wire_od := 2 * wire_r
	layer := 0
	layer_wire := n
	if ow > 0 { 
		layer = n / ow
		layer_wire = n % ow 
	}
	r = loop_r + wire_od * float64(layer)
	x = wire_od * float64(layer_wire)
	return
}

// return turns in first choke donut
// -inputs-
// o = donuts
// N = total turns
// -outputs-
// on = donut turns
func choke_on(o, N int) (on int) {
	on = 1 + (N - 1) / o
	return
}

// return r & x for loop n in multi-donut choke
// wires are laid down per multi-layer donut, from donut to donut
// if ow = 0 do single layer donuts
// if oo > 0 do axial (in-line) donuts
// if oo < 0 do radial (concentric) donuts
// -inputs-
// loop_r = loop radius (m) of first layer @ centerline of conductor
// wire_r = wire radius (m) of OD
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// n = loop number
// N = total turns
// -outputs-
// r = radius of loop (m)
// x = location of loop (m)
// donut = loop donut
func choke_rxo(loop_r, wire_r, oo float64, ow, o, n, N int) (r, x float64, donut int) {
	for {
		on := choke_on(o, N)
		if n < on {
			r, x = donut_rx(loop_r, wire_r, ow, n)
			if oo < 0 {	r -= oo * float64(donut)  // raidal donut offset
			} else { x += oo * float64(donut) }  // axial donut offset
			return
		}
		n -= on
		N -= on
		o--
		donut++
	}
	return
}

// find total turns of a multi-donut choke given wire length
// brain dead exhaustive looping, but seems quick enough and bulletproof
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// wire_r = wire radius (m)
// wire_l = wire length (m)
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// -outputs-
// N = number of turns
func choke_wl_N(loop_r, wire_r, wire_l, oo float64, ow, o int) (N int) {
	n := 0
	err := -1.0
	err_best := -wire_l
	for err < 0 {
		wl := 0.0
		for i:=0; i<n; i++ {
			r, _, _ := choke_rxo(loop_r, wire_r, oo, ow, o, i, n)
			wl += 2 * math.Pi * r
		}
		err = wl - wire_l
		if math.Abs(err) < math.Abs(err_best) { 
			err_best = err
			N = n
		}
		n++
		if n % 500 == 0 { fmt.Print(".") } // show activity
	}
	fmt.Println("wl=>N")
	return
}

// find wire length of a multi-donut choke given total turns
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// N = turns total
// -outputs-
// wire_l = length of wire (m)
func choke_N_wl(loop_r, wire_r, oo float64, ow, o, N int) (wire_l float64) {
	for i:=0; i<N; i++ {
		r, _, _ := choke_rxo(loop_r, wire_r, oo, ow, o, i, N)
		wire_l += math.Pi * 2 * r
	}
	return
}

// find od and width of a multi-donut choke given total turns
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// wire_r = wire radius (m)
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// N = turns total
// -outputs-
// r_max = max radius (m)
// x_max = max x (m)
func choke_rx_max(loop_r, wire_r, oo float64, ow, o, N int) (r_max, x_max float64) {
	for i:=0; i<N; i++ {
		r, x, _ := choke_rxo(loop_r, wire_r, oo, ow, o, i, N)
		if r > r_max { r_max = r }
		if x > x_max { x_max = x }
	}
	return
}

// find inductances of a multi-donut choke given total turns
// total mutual inductance between each turn and every other turn
// total self inductance of each turn
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// wire_r = wire radius (m)
// cond_r = conductor radius (m)
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// N = turns total
// act = show activity
// -outputs-
// l_mut, l_self = inductances (H)
func choke_N_L(loop_r, wire_r, cond_r, oo float64, ow, o, N int, act bool) (l_mut, l_self float64) {
	for i:=0; i<N; i++ {
		r_i, x_i, _ := choke_rxo(loop_r, wire_r, oo, ow, o, i, N)
		l_self += loop_l_self(r_i, cond_r)
		for j:=i+1; j<N; j++ {
			r_j, x_j, _ := choke_rxo(loop_r, wire_r, oo, ow, o, j, N)
			l_mut += loops_l_mut(r_i, r_j, math.Abs(x_i - x_j))
		}
		if act && i % 500 == 0 { fmt.Print(".") } // show activity
	}
	if act { fmt.Println("N=>L") }
	l_mut *= 2
	return
}

// find total turns of a multi-donut choke given total inductance
// -inputs-
// L = target inductance (H)
// loop_r = loop radius (m) @ centerline of conductors
// wire_r = wire radius (m)
// cond_r = conductor radius (m)
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// -outputs-
// N = number of turns
func choke_L_N(L, loop_r, wire_r, cond_r, oo float64, ow, o int) (N int) {
	n := 1
	n_inc := 1
	err_best := L
	init := true
	for n_inc != 0 {
		l_m, l_s := choke_N_L(loop_r, wire_r, cond_r, oo, ow, o, n, false)		
		l_calc := l_m + l_s
		err := l_calc - L
		if math.Abs(err) < math.Abs(err_best) { 
			err_best = err
			N = n
		}
		if err > 0 { init = false }
		if init { n_inc *= 2 } else { n_inc /= 2 }
		if err < 0 { n += n_inc } else { n -= n_inc }
		fmt.Print(".")  // show activity
	}
	fmt.Println("L=>N")
	return
}

// find closest distance between windings of a multi-donut choke 
// for collision purposes
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// wire_r = wire radius (m)
// cond_r = conductor radius (m)
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// N = turns total
// -outputs-
// gap = minimum donuts distance (m)
func choke_gap(loop_r, wire_r, cond_r, oo float64, ow, o, N int) (gap float64) {
	first_f := true
	for i:=0; i<N; i++ {
		r_i, x_i, o_i := choke_rxo(loop_r, wire_r, oo, ow, o, i, N)
		for j:=i+1; j<N; j++ {
			r_j, x_j, o_j := choke_rxo(loop_r, wire_r, oo, ow, o, j, N)
			if (o_i != o_j) {  // different donuts
				r_dist := r_i - r_j
				x_dist := x_i - x_j
				dist := math.Sqrt(r_dist * r_dist + x_dist * x_dist)
				if dist < gap || first_f { 
					gap = dist 
					first_f = false
				}
			}
		}
		if i % 500 == 0 { fmt.Print(".") } // show activity
	}
	fmt.Println("gap")
	gap -= 2 * wire_r
	return
}


//////////
// xfmr //
//////////

// return coupling coefficient k
// l = l_mut / sqrt(l_self1 * l_self2)
func coupling(l_mut, l_self1, l_self2 float64) (k float64) {
	k = l_mut / math.Sqrt(l_self1 * l_self2)
	return
}

// find mutual inductance of a 2 donut xfmr given turns
// -inputs-
// loop_r, loop2_r = loop radius (m) @ centerline of conductors
// wire_r, wire2_r = wire radius (m)
// cond_r, cond2_r = conductor radius (m)
// ow, ow2 = donut width (turns per layer)
// oo = donut offset (m)
// N, N2 = turns
// -outputs-
// l_mut = mutual inductance (H)
func xfmr_N_L(loop_r, loop2_r, wire_r, wire2_r, cond_r, cond2_r, oo float64, ow, ow2, N, N2 int) (l_mut float64) {
	for i:=0; i<N; i++ {
		r_i, x_i := donut_rx(loop_r, wire_r, ow, i)
		for j:=0; j<N2; j++ {
			r_j, x_j := donut_rx(loop2_r, wire2_r, ow2, j)
			l_mut += loops_l_mut(r_i, r_j, math.Abs(x_i - (x_j + oo)))
		}
	}
	l_mut *= 2
	return
}

// find closest distance between windings of a 2 donut xfmr
// for collision purposes
// -inputs-
// loop_r, loop2_r = loop radius (m) @ centerline of conductors
// wire_r, wire2_r = wire radius (m)
// cond_r, cond2_r = conductor radius (m)
// ow, ow2 = donut width (turns per layer)
// oo = donut offset (m)
// N, N2 = turns
// -outputs-
// gap = minimum donuts distance (m)
func xfmr_gap(loop_r, loop2_r, wire_r, wire2_r, cond_r, cond2_r, oo float64, ow, ow2, N, N2 int) (gap float64) {
	first_f := true
	for i:=0; i<N; i++ {
		r_i, x_i := donut_rx(loop_r, wire_r, ow, i)
		for j:=0; j<N2; j++ {
			r_j, x_j := donut_rx(loop2_r, wire2_r, ow2, j)
			r_dist := r_i - r_j
			x_dist := x_i - (x_j + oo)
			dist := math.Sqrt(r_dist * r_dist + x_dist * x_dist)
			if dist < gap || first_f { 
				gap = dist 
				first_f = false
			}
		}
	}
	gap -= wire_r + wire2_r
	return
}


//////////////////////
// skin & proximity //
//////////////////////

// return skin depth given frequency
// skin depth = sqrt(rho / (pi * freq * mu_0))
func skin_delta(f_hz float64) (float64) {
	return math.Sqrt(CU_RHO / (math.Pi * f_hz * MU0))
}

// return skin factor given copper rad & frequency
// Alan Payne approximation:
// skin = 1 / 1- e^–x, where x = 3.96/(d/δ) + 3.8/(d/δ)^2 + 18/(d/δ)^3
// skin is DCR muliplier
func skin_factor(cond_r, f_hz float64) (float64) {
	d_ratio := 2 * cond_r / skin_delta(f_hz)
	x := 3.96 / d_ratio + 3.8 / (d_ratio * d_ratio) + 18 / (d_ratio * d_ratio * d_ratio)
	return 1 / (1 - math.Exp(-x))
}

// return proximity factor for two wires adjacent to each other
// Alan Payne approximation:
// prox = 7.8 * (d/p)^2 * e^-x / (1 + 18 * e^-x)
// d = wire diameter (m)
// x = 10.3 * delta / d
// p = pitch (wire distance center to center) (m)
// prox + 1 is ACR muliplier
func prox_2_wire(cond_r, dist, f_hz float64) (float64) {
	d := 2 * cond_r
	x := 10.3 * skin_delta(f_hz) / d
	expx := math.Exp(-x)
	return 7.8 * math.Pow(d / dist, 2) * expx / (1 + 18 * expx)
}

// return proximity factor for two coaxial wire loops
// my cludge is scaling the influences based on loop radius ratios
func prox_2_loop(cond_r, loop_r, loop2_r, loop_x, loop2_x, f_hz float64) (float64) {
	r_dist := loop_r - loop2_r
	x_dist := loop_x - loop2_x
	dist := math.Sqrt(r_dist * r_dist + x_dist * x_dist)
	return (loop2_r / loop_r) * prox_2_wire(cond_r, dist, f_hz)
}

// find resistance factor of choke due to proximity effect
// -inputs-
// loop_r = loop radius (m) @ centerline of conductors
// wire_r = wire radius (m)
// cond_r = conductor radius (m)
// o = donuts
// ow = donut width (turns per layer)
// oo = donut offset (m)
// N = turns total
// f_hz = frequency (Hz)
// -outputs-
// prox = proximity factor (ACR multiplier)
func choke_prox(loop_r, wire_r, cond_r, oo, f_hz float64, ow, o, N int) (prox float64) {
	r_total := 0.0;
	for i:=0; i<N; i++ {
		r_i, x_i, _ := choke_rxo(loop_r, wire_r, oo, ow, o, i, N)
		r_total += r_i
		for j:=0; j<N; j++ {
			r_j, x_j, _ := choke_rxo(loop_r, wire_r, oo, ow, o, j, N)
			if (i != j) {  // different wires
				prox += r_i * prox_2_loop(cond_r, r_i, r_j, x_i, x_j, f_hz)  // scale by r_i
			}
		}
	}
	prox /= r_total  // scale by r total
	prox += 1
	return
}
